<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\TherapistEducationsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\TherapistEducations\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\TherapistEducationsImport;
use App\Models\TherapistEducation;
use App\Models\Therapist;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class TherapistEducationsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist.add_education');
        $this->middleware('permission:therapist.add_education', ['only' => ['store']]);
        $this->middleware('permission:therapist.add_education', ['only' => ['update']]);
        $this->middleware('permission:therapist.add_education', ['only' => ['destroy']]);
        $this->middleware('permission:therapist.add_education', ['only' => ['export']]);
        $this->middleware('permission:therapist.add_education', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null, $therapist)
    {
        if ($req != null) {
            $therapist_educations =  $therapist->therapist_educations();
            if ($req->trash && $req->trash == 'with') {
                $therapist_educations =  $therapist_educations->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_educations =  $therapist_educations->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_educations = $therapist_educations->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_educations = $therapist_educations->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_educations = $therapist_educations->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_educations = $therapist_educations->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_educations = $therapist_educations->get();
                return $therapist_educations;
            }
            $therapist_educations = $therapist_educations->get();
            return $therapist_educations;
        }
        $therapist_educations = $therapist->therapist_educations()->withAll()->orderBy('id', 'desc')->get();
        return $therapist_educations;
    }


    /*********View All TherapistEducations  ***********/
    public function index(Request $request, Therapist $therapist)
    {
        $therapist_educations = $this->getter($request, null, $therapist);
        return view('super_admins.therapists.therapist_educations.index', compact('therapist_educations', 'therapist'));
    }

    /*********View Create Form of TherapistEducation  ***********/
    public function create(Therapist $therapist)
    {
        return view('super_admins.therapists.therapist_educations.create', compact('therapist'));
    }

    /*********Store TherapistEducation  ***********/
    public function store(CreateRequest $request, Therapist $therapist)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $data = $request->all();
            $data['image'] = uploadFile($request, 'file', 'therapist_educations');
            $therapist_education = $therapist->therapist_educations()->create($data);
            $therapist_education = $therapist->therapist_educations()->withAll()->find($therapist_education->id);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_educations.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_educations.index', $therapist->id)->with('message', 'Education Created Successfully')->with('message_type', 'success');
    }

    /*********View TherapistEducation  ***********/
    public function show(Therapist $therapist, TherapistEducation $therapist_education)
    {
        if ($therapist->id != $therapist_education->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_educations.show', compact('therapist_education', 'therapist'));
    }

    /*********View Edit Form of TherapistEducation  ***********/
    public function edit(Therapist $therapist, TherapistEducation $therapist_education)
    {
        if ($therapist->id != $therapist_education->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapists.therapist_educations.edit', compact('therapist_education', 'therapist'));
    }

    /*********Update TherapistEducation  ***********/
    public function update(CreateRequest $request, Therapist $therapist, TherapistEducation $therapist_education)
    {
        if ($therapist->id != $therapist_education->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            $data = $request->all();
            if ($request->file) {
                $data['image'] = uploadFile($request, 'file', 'therapist_educations', $therapist_education->image);
            } else {
                $data['image'] = $therapist_education->image;
            }
            $therapist_education->update($data);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_educations.index', $therapist->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_educations.index', $therapist->id)->with('message', 'TherapistEducation Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_educations = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_educations." . $extension;
        return Excel::download(new TherapistEducationsExport($therapist_educations), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new TherapistEducationsImport, $file);
        return redirect()->back()->with('message', 'TherapistEducation Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE TherapistEducation ***********/
    public function destroy(Therapist $therapist, TherapistEducation $therapist_education)
    {
        if ($therapist->id != $therapist_education->therapist_id) {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
        $therapist_education->delete();
        return redirect()->back()->with('message', 'TherapistEducation Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE TherapistEducation ***********/
    public function destroyPermanently(Request $request, Therapist $therapist, $therapist_education)
    {
        $therapist_education = TherapistEducation::withTrashed()->find($therapist_education);
        if ($therapist_education) {
            if ($therapist_education->trashed()) {
                if ($therapist_education->image && file_exists(public_path($therapist_education->image))) {
                    unlink(public_path($therapist_education->image));
                }
                $therapist_education->forceDelete();
                return redirect()->back()->with('message', 'TherapistEducation Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'TherapistEducation is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore TherapistEducation***********/
    public function restore(Request $request, Therapist $therapist, $therapist_education)
    {
        $therapist_education = TherapistEducation::withTrashed()->find($therapist_education);
        if ($therapist_education->trashed()) {
            $therapist_education->restore();
            return redirect()->back()->with('message', 'TherapistEducation Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'TherapistEducation Not Found')->with('message_type', 'error');
        }
    }
}
